import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.time.LocalDate;
import java.sql.Date;
import java.sql.Connection;

public class OrderDB {

    private final Connection connection;

    public OrderDB(Connection connection) {
        this.connection = connection;
    }

    public void create(int customerId, int bookId, int orderQty, double bookPrice, LocalDate orderDate) throws DBException {
        try {
            // Begin transaction
            connection.setAutoCommit(false);

            // Check inventory
            int inventoryQty = getInventoryQuantity(bookId);
            if (inventoryQty < orderQty) {
                throw new DBException("Insufficient inventory");
            }

            // Insert into Orders and get the generated order ID
            int orderId = insertOrder(customerId, orderQty, bookPrice, orderDate);

            // Insert into OrderDetails
            insertOrderDetails(orderId, bookId, orderQty, bookPrice);

            // Update inventory
            updateInventory(orderQty, bookId);

            // Commit the transaction
            connection.commit();
        } catch (SQLException e) {
            rollbackTransaction();
            throw new DBException("Database error: " + e.getMessage());
        } finally {
            setAutoCommitTrue();
        }
    }

    private int getInventoryQuantity(int bookId) throws SQLException, DBException {
        var sql = "SELECT Qty FROM Inventories WHERE BookId = ?";
        try (var stmt = connection.prepareStatement(sql)) {
            stmt.setInt(1, bookId);
            try (var rs = stmt.executeQuery()) {
                if (rs.next()) {
                    return rs.getInt("Qty");
                } else {
                    throw new DBException("Book not found in inventory");
                }
            }
        }
    }

    private int insertOrder(int customerId, int quantity, double price, LocalDate orderDate) throws SQLException, DBException {
        var sql = "INSERT INTO Orders (OrderDate, CustomerId, TotalAmount) VALUES (?, ?, ?)";
        try (var stmt = connection.prepareStatement(sql, PreparedStatement.RETURN_GENERATED_KEYS)) {
            stmt.setDate(1, Date.valueOf(orderDate));
            stmt.setInt(2, customerId);
            stmt.setDouble(3, price * quantity);
            stmt.executeUpdate();

            try (var rs = stmt.getGeneratedKeys()) {
                if (rs.next()) {
                    return rs.getInt(1);
                } else {
                    throw new DBException("Failed to retrieve order ID.");
                }
            }
        }
    }

    private void insertOrderDetails(int orderId, int bookId, int quantity, double price) throws SQLException {
        var sql = "INSERT INTO OrderDetails (OrderId, BookId, Quantity, Price) VALUES (?, ?, ?, ?)";

        try (var stmt = connection.prepareStatement(sql)) {
            stmt.setInt(1, orderId);
            stmt.setInt(2, bookId);
            stmt.setInt(3, quantity);
            stmt.setDouble(4, price);
            stmt.executeUpdate();
        }
    }

    private void updateInventory(int quantity, int bookId) throws SQLException {
        var sql = "UPDATE Inventories SET Qty = Qty - ? WHERE BookId = ?";

        try (var stmt = connection.prepareStatement(sql)) {
            stmt.setInt(1, quantity);
            stmt.setInt(2, bookId);
            stmt.executeUpdate();
        }
    }

    private void rollbackTransaction() throws DBException {
        try {
            connection.rollback();
        } catch (SQLException e) {
            throw new DBException("Error rolling back transaction: " + e.getMessage());
        }
    }

    private void setAutoCommitTrue() throws DBException {
        try {
            connection.setAutoCommit(true);
        } catch (SQLException e) {
            throw new DBException("Error setting auto-commit: " + e.getMessage());
        }
    }
}
